// FullTextView.mc (Final version with Circular Clipping and Smooth Scrolling)

using Toybox.WatchUi as Ui;
using Toybox.Graphics as Gfx;
using Toybox.Lang as Lang;
using Toybox.System as Sys;
using Toybox.Math as Math;
using Toybox.Application as App;

class FullTextView extends Ui.View {
    var messageBuffer;
    var screenWidth;
    var screenHeight;
    var screenRadius;
    var totalContentHeight;   // Height of all wrapped text
    var font;                 // Store font
    var lines;                // Store wrapped lines after calculation

    function initialize() { 
        View.initialize();
        screenWidth = Sys.getDeviceSettings().screenWidth;
        screenHeight = Sys.getDeviceSettings().screenHeight;
        screenRadius = screenWidth / 2.0;

        totalContentHeight = 0;
        font = Gfx.FONT_SMALL;
        lines = [];
        

        
        // Initial text wrapping is called in onUpdate to ensure dc is available,
        // but we'll pre-check it here if possible, and definitely in onUpdate.
    }
    
    // Helper function to calculate the available width at a given y-coordinate
    function getClippingWidth(y) {
        var y_center = y - screenRadius; 
        var radiusSq = screenRadius * screenRadius;
        var yCenterSq = y_center * y_center;
        
        if (yCenterSq > radiusSq) {
            return 0;
        }
        
        var x_half = Math.sqrt(radiusSq - yCenterSq);
        var clipWidth = x_half.toNumber() * 2;
        
        // Return clipWidth with a minimum padding of 10
        return clipWidth > 10 ? clipWidth - 10 : 0;
    }

    // Calculate and store the wrapped lines
    function wrapText(dc) {
        lines = [];
        var words = split(messageBuffer, " ");
        var currentLine = "";
        var lineHeight = dc.getFontHeight(font);
        var currentY = 5; // Starting Y position for calculation

        for (var i = 0; i < words.size(); i++) {
            var word = words[i];
            var testLine = currentLine.length() > 0 ? currentLine + " " + word : word;
            
            // Calculate the max available width for the current Y position
            // Use line center for a more accurate width calculation
            var maxLineWidth = getClippingWidth(currentY + (lineHeight / 2)); 
            
            if (maxLineWidth <= 0) {
                 // Skip if no horizontal space is available
                 currentLine = word;
                 currentY += lineHeight;
                 continue;
            }

            if (dc.getTextDimensions(testLine, font)[0] < maxLineWidth) {
                currentLine = testLine;
            } else {
                lines.add(currentLine);
                currentY += lineHeight;
                currentLine = word;
            }
        }
        lines.add(currentLine); 
        totalContentHeight = (lines.size() * lineHeight) + 10;

    }


    function onUpdate(dc) {
        wrapText(dc); // Wrap text every update to ensure correct sizing
        
        dc.setColor(Gfx.COLOR_WHITE, Gfx.COLOR_BLACK);
        dc.clear();
        
        var lineHeight = dc.getFontHeight(font);
        var y_start = 5; 
        var lineY = y_start;
        
        // === Drawing Logic (with clipping) ===
        for (var i = 0; i < lines.size(); i++) {
            var line = lines[i];
            
            // Get the clipped width and horizontal offset for this line
            var clippedWidth = getClippingWidth(lineY + (lineHeight / 2)); 
            var x_start = (screenWidth - clippedWidth) / 2; 

            // Check if the line is vertically visible
            if (lineY > screenHeight) {
                break;
            }
            if (lineY + lineHeight < 0) {
                lineY += lineHeight;
                continue;
            }

            // Draw the line, centered within the clipped area
            dc.drawText(
                x_start + (clippedWidth / 2),
                lineY,
                font,
                line,
                Gfx.TEXT_JUSTIFY_CENTER
            );
            
            lineY += lineHeight;
        }

    }

    // Utility function (from your original code)
    function split(s, sep) {
        var tokens = [];
        var found = s.find(sep);

        while (found != null) {
            var token = s.substring(0, found);
            tokens.add(token);
            s = s.substring(found + sep.length(), s.length());
            found = s.find(sep);
        }

        tokens.add(s);

        return tokens;
    }
}