using Toybox.WatchUi as Ui;
using Toybox.System as Sys;
using Toybox.Application as App;
using Toybox.Timer as Timer;

class EditDelegate extends Ui.InputDelegate {
    var keyboardLayout;
    var messageBuffer;
    var view; // Reference to the view
    var eraseTimer;
    var isErasing;
    
    function initialize(layout, viewInstance) {
        InputDelegate.initialize();
        keyboardLayout = layout;
        view = viewInstance; // Store the view instance
        messageBuffer = view.textView.messageBuffer;
        eraseTimer = new Timer.Timer();
        isErasing = false;
    }

    function eraseCharacter() {
        if (messageBuffer.length() > 0) {
            messageBuffer = messageBuffer.substring(0, messageBuffer.length()-1);
            view.updateBuffer(messageBuffer);
        }
    }

    function eraseRepeatedly() {
        if (isErasing) {
            eraseCharacter();
        } else {
            // Safety measure: stop the timer if the flag is false
            eraseTimer.stop();
        }
    }

    function onHold(clickEvent) {
        var coords = clickEvent.getCoordinates();
        var holdX = coords[0];
        var holdY = coords[1];

        // Check if the hold started on the backspace bitmap
        if (holdX >= view.bmpX && holdX <= view.bmpX + view.bmpSize &&
            holdY >= view.bmpY && holdY <= view.bmpY + view.bmpSize) {

            if (!isErasing) {
                isErasing = true;
                
                // Start the continuous erase timer (e.g., every 150ms)
                eraseTimer.start(method(:eraseRepeatedly), 50, true);
                return true;
            }
            return true; // Event was handled
        }
        return false;
    }

    function onRelease(clickEvent) {
        // Stop the timer regardless of where the release happened, 
        // as long as the erasing process was active.
        if (isErasing) {
            eraseTimer.stop();
            isErasing = false;
            return true;
        }
        return false;
    }

    function onTap(clickEvent) {
        var coords = clickEvent.getCoordinates();
        var tapX = coords[0];
        var tapY = coords[1];


        // Check if tap was on the backspace bitmap
        if (tapX >= view.bmpX && tapX <= view.bmpX + view.bmpSize &&
            tapY >= view.bmpY && tapY <= view.bmpY + view.bmpSize) {

            // backspace (only execute on a tap if it's NOT already holding)
            if (!isErasing) {
                 eraseCharacter(); // Perform single erase for a quick tap
            }
            return true; // Handled the tap on the backspace area
        }


        // Check if tap was on a key
        for (var i = 0; i < keyboardLayout.keys.size(); i++) {
            var key = keyboardLayout.keys[i];
            if (tapX >= key.x && tapX <= key.x + key.width &&
                tapY >= key.y && tapY <= key.y + key.height) {
                var key_val = key.label[keyboardLayout.keyboardState];

                if (key_val.equals("П") or key_val.equals("space") or key_val.equals("<-")){
                    if (key_val.equals("П")) {
                        keyboardLayout.keyboardState += 1;
                        keyboardLayout.keyboardState = keyboardLayout.keyboardState%3;
                    }
                    if (key_val.equals("space")){
                        messageBuffer += " ";
                    }
                    if (key_val.equals("<-")) {
                        messageBuffer += "\n";
                    }
                
                    view.updateBuffer(messageBuffer);
                    return true;
                } else {
                    // Add the character to the buffer
                    messageBuffer += key_val;
                    view.updateBuffer(messageBuffer); // Call the view's update method
                    keyboardLayout.keyboardState = 0;
                    return true; // Event was handled
                }
            }
        }
        return false; // Event was not handled by a key
    }
}